#include "serial.h"

// Open a serial communication on the specified port.
// Sets the data rate in bits per second (baud) for serial data transmission.
void serial_begin(const char *serial_port, int serial_speed){

	struct termios options;
	int speed;

	switch(serial_speed){
		case     50:	speed =     B50 ; break ;
		case     75:	speed =     B75 ; break ;
		case    110:	speed =    B110 ; break ;
		case    134:	speed =    B134 ; break ;
		case    150:	speed =    B150 ; break ;
		case    200:	speed =    B200 ; break ;
		case    300:	speed =    B300 ; break ;
		case    600:	speed =    B600 ; break ;
		case   1200:	speed =   B1200 ; break ;
		case   1800:	speed =   B1800 ; break ;
		case   2400:	speed =   B2400 ; break ;
		case   9600:	speed =   B9600 ; break ;
		case  19200:	speed =  B19200 ; break ;
		case  38400:	speed =  B38400 ; break ;
		case  57600:	speed =  B57600 ; break ;
		case 115200:	speed = B115200 ; break ;
		default:	speed = B230400 ; break ;
			
	}


	if ((sd = open(serial_port, O_RDWR | O_NOCTTY | O_NDELAY | O_NONBLOCK)) == -1){
		fprintf(stderr,"Unable to open the serial port %s\n", serial_port);
		exit(1);
	}
    
	fcntl (sd, F_SETFL, O_RDWR) ;
    
	tcgetattr(sd, &options);
	cfmakeraw(&options);
	cfsetispeed (&options, speed);
	cfsetospeed (&options, speed);

	options.c_cflag |= (CLOCAL | CREAD);
	options.c_cflag &= ~PARENB;
	options.c_cflag &= ~CSTOPB;
	options.c_cflag &= ~CSIZE;
	options.c_cflag |= CS8;
	options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
	options.c_oflag &= ~OPOST;

	tcsetattr (sd, TCSANOW, &options);

	ioctl (sd, TIOCMGET, &status);

	status |= TIOCM_DTR;
	status |= TIOCM_RTS;

	ioctl (sd, TIOCMSET, &status);
	
	usleep (10000);
    
}

//Prints one byte to the serial port.
void serial_print (unsigned char message){
	write(sd,&message,1);
}

//Prints multiple bytes to the serial port.
void serial_print_string(const char *message){
	write(sd,message,strlen(message));
}

// Get the numberof bytes (characters) available for reading from 
// the serial port.
// Return: number of bytes available to read */
int serial_available(){
    int nbytes = 0;
    if (ioctl(sd, FIONREAD, &nbytes) < 0)  {
		fprintf(stderr, "Failed to get byte count on serial.\n");
        exit(-1);
    }
    return nbytes;
}

// Reads 1 byte of incoming serial data
// Returns: first byte of incoming serial data available */
unsigned char serial_read_byte() {
	read(sd,&c,1);
    return c;
}

// Returns the next byte (character) of incoming serial data without removing it
// from the internal serial buffer.
unsigned char serial_peek(){
    //We obtain a pointer to FILE structure from the file descriptor sd
    FILE * f = fdopen(sd,"r+");
    //With a pointer to FILE we can do getc and ungetc
    c = getc(f);
    ungetc(c, f);
    return c;
}

// Remove any data remaining on the serial buffer
void serial_flush(){
    while(serial_available()){
        read(sd,&c,1);
    }
}

//Close serial communication
void serial_end(){
	close(sd);
}
