/*
Increasing the number of max open files in system is needed for keeping many 
connections opened.
We need to increase the hard and soft limits for opened files. 
For example we can do this for all system users changing them to:

  *       soft    nofile  1000000
  *       hard    nofile  1000000

This will set the hard limit to 1000000 files and the soft to 500000.

(On ubuntu this can be changed on the file /etc/security/limits.conf)
*/

var net       = require('net');
var mqtt      = require('mqtt');
var cluster   = require('cluster');
var numCPUs   = require('os').cpus().length;
var xmlParser = require('xml2js').parseString;

var privateKey  = __dirname + '/key.csr.server1.pem';
var certificate = __dirname + '/crt.server1.pem';


// Seconds to wait for ACK from appliance since last message received. 
// In this period of time only ACK message will be accepted, 
// everything else will be dropped.
var WAIT_FOR_ACK = 15;

// Helper function to extract commands from an xml list of commands and format
// them in a simpler way for the Linux board. If the xml is not as expected this
// function will return it with no changes.
// The format for the Linux board will be:
// COMMAND_NAME[ATTR:VAL,ATTR:VAL, ...];COMMAND_NAME[ATTR:VAL,ATTR:VAL]; ...
// ommiting the type attribute and atributes with no value.
// The commands formated this way will be used as payload in an MQTT frame.
function xml2command(xml){
  try{
    var commandsString = "";
    xmlParser(xml, function(err,result){
      var jsonCommands = result.Appliance.COMMAND;
      for(var i in jsonCommands){
        for (var key in jsonCommands[i]) {
          if(key == "Name"){
            commandsString += jsonCommands[i][key] + '[';
          }else if(jsonCommands[i][key] != '' && key != 'Type'){
            commandsString += key + ':' + jsonCommands[i][key] + ','
          }
        }
        commandsString = commandsString.slice(0,-1);
        commandsString += '];';
      }
      commandsString = commandsString.slice(0,-1);
    });
  }catch (exception){
    console.log("Exception parsing xml. Returning unmodified xml");
    return xml;
  }
  return commandsString
}

// Helper function to convert command received from appliance to XML.
// Format of command received from appliance is as follows:
// COMMAND_NAME:STATUS. For example TEMPERATURE_VALUE:Done
function command2xml(command){
  try{
    var parts = command.toString().split(":");
    var xml = '<?xml version="1.0"?><Appliance xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
    xml += '<COMMAND><Name>'+parts[0]+'</Name><Status>'+parts[1]+'</Status></COMMAND></Appliance>';
  }catch(exception){return command;}
    return xml;
  }



/*******************************************************************************
  MASTER PROCESS
*******************************************************************************/
if (cluster.isMaster){

  var workers = {};
  var DIsocket;
  // Fork workers.
  for (var i=0; i<numCPUs;i++){
    forkWorker();
  }

  cluster.on('death', function(worker){
    console.log('Worker '+ worker.process.pid + ' died');
    // Remove the worker with its associated IDs from workers dictionary
    for (var i in workers){
      if(workers[i] == worker)  delete workers[i];
    }
      // Create a new worker
      forkWorker();
  });

  // Master process have a tcp server for communicating with DI.
  var server = net.createServer(function (DI) {
    DIsocket = DI;
    var id = DI.remoteAddress + ':' + DI.remotePort;
    console.log('New connection from DI ' + id);
    
    // DATA RECEIVED FROM DI (THIS WILL BE MODIFIED, BECAUSE DI WILL BE INTEGRATED
    // WITH DI-CM, SO THE INFO FROM DI WILL NOT COME FROM A SOCKET).
    DI.on('data', function(data) {
      console.log('COMMANDS FROM DI TO APPLIANCE:');
      console.log(data.toString('utf8'));
      console.log('COMMANDS CONVERTED FOR SENDING TO APPLIANCE:');
      var convertedCommands = xml2command(data.toString('utf8'));
      console.log(convertedCommands+'\n');

      // We must know the appliance ID to which we want to send the commands
      // For the moment just to debug we will send them to the appliance ID = 90:A2:DA:F0:04:E6
        try{
          var applianceID="90:A2:DA:F2:03:61";
          workers[applianceID].send({ID:applianceID,commands:convertedCommands});
        }catch(e){console.log('Error sending data to worker. Maybe it is dead');}

    });

    DI.on('end', function() {
      console.log('DI connection ended');
      DIsocket = null;
      DI.destroy();
    });
  });

function forkWorker(){
  worker = cluster.fork();

  // Message received from a worker
  worker.on('message', function(msg){
    //Appliance login
    if (msg.login){
      workers[msg.ID] = cluster.workers[msg.workerID];  

      // Appliance status update
    }else if(msg.stat_update){
      
      console.log("STATUS UPDATE RECEIVED");
      
    // ACK received from appliance
    }else if (msg.ACK){
      console.log("ACK received");

    // Appliance disconnected
    }else if (msg.connection == "disconnect"){
      // An appliance has been disconnected
      // Remove association ID-worker from workers dictionary
      delete workers[msg.ID];
    }
  });
}

server.listen(8001);
/******************************************************************************/


/*******************************************************************************
  WORKER PROCESSES
*******************************************************************************/
}else{
  // Worker processes have a tcp server listening appliances.
  var sockets = {};
  var timerAndACK = {};

  function cleanConnection(client){
    delete sockets[client.id];
    clearTimeout(timerAndACK[client.id].timer);
    delete timerAndACK[client.id];
    //Notify master process that the appliance is disconnected
    process.send({connection:"disconnect",ID:client.id});
    console.log('Appliance ' + client.id + ' ended connection');
  }

  function updateApplianceTimer(client){
    clearTimeout(timerAndACK[client.id].timer);
    timerAndACK[client.id].timer = setTimeout(function(){
      console.log("TIMEOUT");
      client.stream.end();
      try{cleanConnection(client);}catch(e){console.log("Imposible to clean connection. It was previously cleaned");}
    },timerAndACK[client.id].keepalive * 1500);
  }

  var server = mqtt.createSecureServer(privateKey,certificate,
    function (client) {
        client.on('connect', function(packet){
          console.log('A new connection was made');
          // Set the client.id field with the client ID from the CONNECT frame
          client.id = packet.clientId;

          //Configure a timer with the "grace" time (1,5 times the keepalive).
          //If we dont receive a message from the client within one and a half times
          //the keepalive period we disconnect the client. 
          var applianceTimer = setTimeout(function(){
            //This should cause a client.on(close) event. There is the cleanup.
            client.stream.end();
          },packet.keepalive * 1500);

          timerAndACK[client.id] = {keepalive:packet.keepalive,
                                    waitingACK:0,
                                    timer:applianceTimer
                                    };

          //Notify the appliance that the connection was successful
          client.connack({returnCode: 0});
        });

        // DATA RECEIVED FROM APPLIANCE
        client.on('publish', function(packet) {
          //var applianceID = data.ID;

          if(packet.topic == 'login'){
            var data = JSON.parse(packet.payload);
            //sockets[data.ID] = client;
            sockets[client.id] = client;
            // Send to master process for delivery to DI
            //process.send({login:"login",ID:data.ID,workerID:cluster.worker.id});
            process.send({login:"login",ID:client.id,workerID:cluster.worker.id});
            //console.log('Appliance logged:', data.ID);
            console.log('Appliance logged: ', client.id, ' login: ', data.login, ' pass: ', data.pass);
            // Send publish ack to appliance. Is expecting it when QOS > 0
            client.puback({messageId:packet.messageId});
          }else if (packet.topic == 'status_update'){
            // send to the master a message with the status update.
            // The massage incomming from the appliance will be as follows:
            // COMMAND_NAME:STATUS. For example: TEMPERATURE_VALUE:Done
            // This format is previously converted to XML.
	          console.log('STATUS_UPDATE RECEIVED FROM APPLIANCE: \n' + packet.payload + '\n');
            //data = command2xml(data.commands);
	          //console.log('DATA RECEIVED FROM APPLIANCE XML CONVERTED: \n' + data + '\n');
            process.send({ID:client.id,stat_update:packet.payload});
            // Send publish ack to appliance. Is expecting it when QOS > 0
            client.puback({messageId:packet.messageId});
          }else if(packet.topic == 'info'){
            console.log('INFO FROM ',client.id,': ', packet.payload);
            process.send({info:packet.payload,ID:client.id});
          }

          //Reset appliance timer
          updateApplianceTimer(client);

        });

        client.on('puback',function(packet){
          try{
            timerAndACK[client.id].waitingACK=0;
            //Reset appliance timer
            updateApplianceTimer(client);
          }catch(e){
            console.log('Impossible to update timer for appliance. Maybe it was disconnected for inactivity \n');
          }
        });

      	client.on('pingreq', function(packet) {
      		client.pingresp();
          //Reset appliance timer
          try{
            updateApplianceTimer(client);
          }catch(e){
            console.log('Impossible to update timer for appliance. Maybe it was disconnected for inactivity \n');
          }
        });

        client.on('disconnect', function(packet){
          console.log("EVENT DISCONNECT");
        });

        client.on('close', function() {
          try{cleanConnection(client);}catch(e){console.log("Imposible to clean connection. It was previously cleaned");}
        });

        client.on('error', function(err) {
          client.stream.end();
          console.log('error!');
        });

}).listen(1883);

  // Message received from master
  process.on('message', function(msg){
    if(msg.commands){
      // Search for the correct socket (based on ID field of message) for sending commands.
      if (sockets[msg.ID] != null){
        try{
          // Send commands to appliance if we are not waiting ACK from previous message
          // Discriminate between different types of messages, changing the MQTT topic
          if(timerAndACK[msg.ID].waitingACK == 0){
            sockets[msg.ID].publish({topic: 'commands', payload: msg.commands,qos:1,messageId:1});

            timerAndACK[msg.ID].waitingACK = 1;
            console.log('COMMANDS SENT TO APPLIANCE\n');
          }else{
            //NOTIFY MASTER THAT WE HAVEN'T SEND THE MESSAGE BECAUSE WE ARE WAITING ACK FROM PREVIOUS MESSAGE
          }
        }catch(e){
          console.log('Error sending commands to appliance '+msg.ID+'. Maybe connection is closed')
        }
      }else{
        console.log('Appliance is disconnected');
      }
    }
  });
}
/******************************************************************************/
